/*******************************************************************************
 * This file is part of SWIFT.
 * Copyright (c) 2024 Jonathan Davies (j.j.davies@ljmu.ac.uk)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#ifndef SWIFT_BASIC_SINK_PROPERTIES_H
#define SWIFT_BASIC_SINK_PROPERTIES_H

/* Local header */
#include "feedback_properties.h"
#include "parser.h"

/**
 * @brief Properties of sink in the Default model.
 */
struct sink_props {

  /* ----- Basic neighbour search properties ------ */

  /*! Resolution parameter */
  float eta_neighbours;

  /*! Target weightd number of neighbours (for info only)*/
  float target_neighbours;

  /*! Smoothing length tolerance */
  float h_tolerance;

  /*! Tolerance on neighbour number  (for info only)*/
  float delta_neighbours;

  /*! Maximal number of iterations to converge h */
  int max_smoothing_iterations;

  /*! Maximal change of h over one time-step */
  float log_max_h_change;

  /*! Are we using a fixed cutoff radius? (all smoothing length calculations are
   * disabled if so) */
  char use_fixed_r_cut;

  /* Use nibbling rather than swallowing for gas? */
  float use_nibbling;

  /* Gas mass below which sinks will not nibble. */
  float min_gas_mass_for_nibbling;
};

/**
 * @brief Initialise the sink properties from the parameter file.
 *
 * @param sp The #sink_props.
 * @param phys_const The physical constants in the internal unit system.
 * @param us The internal unit system.
 * @param params The parsed parameters.
 * @param cosmo The cosmological model.
 * @param with_feedback Are we running with feedback?
 */
INLINE static void sink_props_init(
    struct sink_props *sp, struct feedback_props *fp,
    const struct phys_const *phys_const, const struct unit_system *us,
    struct swift_params *params, const struct hydro_props *hydro_props,
    const struct cosmology *cosmo, const int with_feedback) {

  /* We don't use a fixed cutoff radius in this model. This property must always
   * be present, as we use it to skip smoothing length iterations in
   * runner_ghost if a fixed cutoff is being used. */
  sp->use_fixed_r_cut = 0;

  /* Read in the basic neighbour search properties or default to the hydro
     ones if the user did not provide any different values */

  /* Kernel properties */
  sp->eta_neighbours = parser_get_opt_param_float(
      params, "BasicSink:resolution_eta", hydro_props->eta_neighbours);

  /* Tolerance for the smoothing length Newton-Raphson scheme */
  sp->h_tolerance = parser_get_opt_param_float(params, "BasicSink:h_tolerance",
                                               hydro_props->h_tolerance);

  /* Get derived properties */
  sp->target_neighbours = pow_dimension(sp->eta_neighbours) * kernel_norm;
  const float delta_eta = sp->eta_neighbours * (1.f + sp->h_tolerance);
  sp->delta_neighbours =
      (pow_dimension(delta_eta) - pow_dimension(sp->eta_neighbours)) *
      kernel_norm;

  /* Number of iterations to converge h */
  sp->max_smoothing_iterations =
      parser_get_opt_param_int(params, "BasicSink:max_ghost_iterations",
                               hydro_props->max_smoothing_iterations);

  /* Time integration properties */
  const float max_volume_change =
      parser_get_opt_param_float(params, "BasicSink:max_volume_change", -1);
  if (max_volume_change == -1)
    sp->log_max_h_change = hydro_props->log_max_h_change;
  else
    sp->log_max_h_change = logf(powf(max_volume_change, hydro_dimension_inv));

  sp->use_nibbling = parser_get_param_int(params, "BasicSink:use_nibbling");
  if (sp->use_nibbling) {
    sp->min_gas_mass_for_nibbling = parser_get_param_float(
        params, "BasicSink:min_gas_mass_for_nibbling_Msun");
    sp->min_gas_mass_for_nibbling *= phys_const->const_solar_mass;
  }
}

/**
 * @brief Write a sink_props struct to the given FILE as a stream of
 * bytes.
 *
 * @param props the sink properties struct
 * @param stream the file stream
 */
INLINE static void sink_struct_dump(const struct sink_props *props,
                                    FILE *stream) {
  restart_write_blocks((void *)props, sizeof(struct sink_props), 1, stream,
                       "sink props", "Sink props");
}

/**
 * @brief Restore a sink_props struct from the given FILE as a stream of
 * bytes.
 *
 * @param props the sink properties struct
 * @param stream the file stream
 */
INLINE static void sink_struct_restore(const struct sink_props *props,
                                       FILE *stream) {
  restart_read_blocks((void *)props, sizeof(struct sink_props), 1, stream, NULL,
                      "Sink props");
}

#endif /* SWIFT_BASIC_SINK_PROPERTIES_H */
