/*******************************************************************************
 * This file is part of SWIFT.
 * Copyright (c) 2012 Pedro Gonnet (pedro.gonnet@durham.ac.uk)
 *                    Matthieu Schaller (schaller@strw.leidenuniv.nl)
 *               2015 Peter W. Draper (p.w.draper@durham.ac.uk)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#ifndef SWIFT_CELL_SINKS_H
#define SWIFT_CELL_SINKS_H

/* Config parameters. */
#include <config.h>

/* Local includes. */
#include "lock.h"
#include "timeline.h"

/**
 * @brief Sinks-related cell variables.
 */
struct cell_sinks {

  /* If we are not using sinks, compact as much of the unecessary variables
     into an anonymous union to save memory in the cell structure. */
#ifdef SINK_NONE
  union {
#endif

    /*! Pointer to the #sink data. */
    struct sink *parts;

    /*! Pointer to the #spart data at rebuild time. */
    struct sink *parts_rebuild;

    /*! Linked list of the tasks computing this cell's sink swallow. */
    struct link *swallow;

    /*! Linked list of the tasks computing this cell's sink do_gas_swallow. */
    struct link *do_gas_swallow;

    /*! Linked list of the tasks computing this cell's sink do_sink_swallow. */
    struct link *do_sink_swallow;

    /*! The drift task for sinks */
    struct task *drift;

    /*! Implicit tasks marking the entry of the sink block of tasks */
    struct task *sink_in;

    /*! The sink ghost task itself */
    struct task *density_ghost;

    /*! Linked list of the tasks computing this cell's sink density. */
    struct link *density;

    /*! Implicit tasks marking the end of sink swallow */
    struct task *sink_ghost1;

    /*! Implicit tasks marking the separation between do_gas_swallow and
     * do_sink_swallow */
    struct task *sink_ghost2;

    /*! Implicit tasks marking the exit of the sink block of tasks */
    struct task *sink_out;

    /*! Task for star formation from sink particles */
    struct task *star_formation_sink;

    /*! Task for sink formation */
    struct task *sink_formation;

    /*! Last (integer) time the cell's sink were drifted forward in time. */
    integertime_t ti_old_part;

    /*! Spin lock for sink formation use. */
    swift_lock_type sink_formation_lock;

    /*! Nr of #sink this cell can hold after addition of new one. */
    int count_total;

    /*! Max smoothing length of active particles in this cell.
     */
    float h_max_active;

    /*! Values of h_max before the drifts, used for sub-cell tasks. */
    float h_max_old;

    /*! Maximum part movement in this cell since last construction. */
    float dx_max_part;

    /*! Values of dx_max before the drifts, used for sub-cell tasks. */
    float dx_max_part_old;

#ifdef SINK_NONE
  };
#endif

  /*! Minimum end of (integer) time step in this cell for sink tasks. */
  integertime_t ti_end_min;

  /*! Maximum beginning of (integer) time step in this cell for sink
   * tasks. */
  integertime_t ti_beg_max;

  /*! Spin lock for various uses (#sink case). */
  swift_lock_type lock;

  /*! Max smoothing length in this cell. */
  float h_max;

  /*! Number of #sink updated in this cell. */
  int updated;

  /*! Is the #sink data of this cell being used in a sub-cell? */
  int hold;

  /*! Nr of #sink in this cell. */
  int count;
};

#endif /* SWIFT_CELL_SINKS_H */
