/*******************************************************************************
 * This file is part of SWIFT.
 * Copyright (c) 2018 Matthieu Schaller (matthieu.schaller@durham.ac.uk)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#ifndef SWIFT_TRACERS_STRUCT_COLIBRE_H
#define SWIFT_TRACERS_STRUCT_COLIBRE_H

/* Local includes */
#include "tracers_triggers.h"

/*! The possible accretion modes every black hole can take. */
enum BH_accretion_modes {
  BH_thick_disc = 0,       /* At low Eddington ratios */
  BH_thin_disc,            /* At moderate Eddington ratios */
  BH_slim_disc,            /* Super-Eddington accretion */
  BH_accretion_modes_count /* Number of possible accretion modes */
};

/**
 * @brief Properties of the tracers stored in the extended particle data.
 */
struct tracers_xpart_data {

  /*! Maximum temperature achieved by this particle */
  float maximum_temperature;

  /*! Cumulative momentum received from stellar winds (physical internal units)
   */
  float stellar_wind_momentum_received;

#if !defined(COOLING_CHIMES)
  /*! Hydrogen fractions */
  float nHI_over_nH;
  float nHII_over_nH;
  float nH2_over_nH;
#endif

  /*! HII region timer */
  float HIIregion_timer_gas;

  /*! Id of the star particle responsible for HII region */
  long long HIIregion_starid;

  /*! Anonymous union for the cosmological non-cosmological runs distinction */
  union {

    /*! Scale-factor at which the maximal temperature was reached */
    float maximum_temperature_scale_factor;

    /*! Time at which the maximal temperature was reached */
    float maximum_temperature_time;
  };

  union {

    /*! Scale-factor at which the particle was last kicked in stellar early
     * feedback */
    float last_stellar_wind_kick_scale_factor;

    /*! Time at which the particle was last kicked in stellar early feedback */
    float last_stellar_wind_kick_time;
  };

  union {

    /*! Scale-factor at which the particle last received energy from type II SNe
     in thermal form */
    float last_SNII_thermal_injection_scale_factor;

    /*! Time at which the particle last received energy from type II SNe in
     * thermal form */
    float last_SNII_thermal_injection_time;
  };

  union {

    /*! Scale-factor at which the particle last received kick from type II SNe
     */
    float last_SNII_kick_scale_factor;

    /*! Time at which the particle last received kick from type II SNe */
    float last_SNII_kick_time;
  };

  union {

    /*! Scale-factor at which the particle last received energy from type Ia SNe
     in thermal form */
    float last_SNIa_thermal_injection_scale_factor;

    /*! Time at which the particle last received energy from type Ia SNe in
     * thermal form */
    float last_SNIa_thermal_injection_time;
  };

  union {

    /*! Scale-factor at which the particle last received energy from AGN */
    float last_AGN_injection_scale_factor;

    /*! Time at which the particle last received energy from AGN */
    float last_AGN_injection_time;
  };

  union {
    /* Last scale factor this particle was kicked as part of jet feedback */
    float last_AGN_jet_feedback_scale_factor;

    /* Last time this particle was kicked as part of jet feedback */
    float last_AGN_jet_feedback_time;
  };

  /*! Averaged SFR over two different time slices */
  float averaged_SFR[num_snapshot_triggers_part];

  /*! Density of the gas at the last SN (type-II or type-Ia) feedback event
   * (physical internal units) */
  float density_at_last_SN_thermal_feedback_event;

  /*! Density of the gas at the last AGN feedback event
   * (physical internal units) */
  float density_at_last_AGN_feedback_event;

  /*! Temperature increase of the gas at the last AGN feedback event
   * (physical internal units) */
  float delta_T_at_last_AGN_feedback_event;

  /*! Total amount of AGN feedback energy received by this particle
   * (physical internal units) */
  float AGN_feedback_energy;

  /*! The energy the particle received the last time it was heated by
   *  AGN feedback (physical internal units) */
  float last_AGN_feedback_energy;

  /*! Counts how many times this particle has been kicked as part of
     jet feedback */
  char hit_by_jet_feedback;

  /*! Total jet feedback energy received by this particle */
  float jet_feedback_energy;

  /*! Kick velocity at last AGN jet event */
  float last_jet_kick_velocity;

  /*! Quantities used for radio emission */
  struct {

    /*! (Physical) volume the particle had when it was last kicked */
    float last_jet_kick_initial_volume;

    /*! Integrals of the shocking rate times volume ratio raised to different
        powers. */
    float shocking_integrals[3];

    /*! Integrals of the energy density (of the magnetic field and CMB,
        respectively), used to compute the minimum of the electron Lorentz
        factor distribution. */
    float gamma_integrals[2];

    /*! Integrals over the above gamma integrals, used to compute the effective
        minimum of the electron Lorentz factor distribution. */
    float gamma_effective_integrals[2];

    /*! The integral of the shocking rate times the scale factor. Used to
       compute the weighted injection scale factor. */
    float weighted_injection_scale_factor;

  } jet_radio_emission;

  /*! The accretion/feedback mode of the BH when this particle was last
   * kicked */
  enum BH_accretion_modes last_jet_kick_accretion_mode;

  /*! The ID of the BH that did the last kick */
  long long last_jet_kick_BH_id;

  /*! Kick velocity in last SNII kinetic feedback event (physical internal
   * units) */
  float last_SNII_kick_velocity;

  /*! The maximal velocity the particle has been kicked with in SNII kinetic
   * feedback events (physical internal units) */
  float max_SNII_kick_velocity;

  /*! The temperature difference applied during the last SNII or SNIa thermal
   * feedback event. */
  float delta_T_at_last_SN_thermal_feedback_event;

  union {
    /*! Last scale factor this particle was part of the ISM. */
    float last_ISM_scale_factor;

    /*! Last time this particle was part of the ISM. */
    float last_ISM_time;
  };

  /*! Minimal smoothing length reached by the black hole */
  float min_h;

  union {

    /* Scale factor when the minimal h was reached */
    float min_h_scale_factor;

    /* Time when the minimal h was reached */
    float min_h_time;
  };

  /*! FOF Halo mass this particle was last in. */
  float last_halo_mass;

  union {

    /* Scale factor when the particle was last in a fof halo */
    float last_in_halo_scale_factor;

    /* Time when the particle as last in a fof halo */
    float last_in_halo_time;
  };
};

/**
 * @brief Properties of the tracers stored in the star particle data.
 *
 * Note: In this model, they are identical to the xpart data.
 */
#define tracers_spart_data tracers_xpart_data

/**
 * @brief Properties of the tracers stored in the black hole particle data.
 */
struct tracers_bpart_data {

  /*! Minimal time-bin reached by this BH */
  timebin_t min_time_bin;

  union {

    /* Scale factor when the minimal time bin was reached */
    float min_time_bin_scale_factor;

    /* Time when the minimal time bin was reached */
    float min_time_bin_time;
  };

  /*! Averaged accretion rate over two different time slices */
  float averaged_accretion_rate[num_snapshot_triggers_bpart];
};

#endif /* SWIFT_TRACERS_STRUCT_COLIBRE_H */
