/*******************************************************************************
 * This file is part of SWIFT.
 * Copyright (c) 2018 Folkert Nobels (nobels@strw.leidenuniv.nl)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#ifndef SWIFT_STAR_FORMATION_EAGLE_IO_H
#define SWIFT_STAR_FORMATION_EAGLE_IO_H

/* Config parameters. */
#include <config.h>

/* Local includes */
#include "io_properties.h"

/**
 * @brief Specifies which s-particle fields to read from a dataset
 *
 * @param sparts The s-particle array.
 * @param list The list of i/o properties to read.
 *
 * @return num_fields The number of i/o fields to read.
 */
INLINE static int star_formation_read_particles(struct spart* sparts,
                                                struct io_props* list) {
  return 0;
}

INLINE static void convert_part_sfr(const struct engine* e,
                                    const struct part* p,
                                    const struct xpart* xp, float* ret) {

  /* The SFR field contains the SFR if > 0 and
   * the (opposite of the) last SF scale-factor (or time) if < 0. */
  ret[0] = (xp->sf_data.SFR >= 0.) ? xp->sf_data.SFR : 0.f;
}

INLINE static void convert_part_last_sf_time(const struct engine* e,
                                             const struct part* p,
                                             const struct xpart* xp,
                                             float* ret) {

  /* The SFR field contains the SFR if > 0 and
   * the (opposite of the) last SF scale-factor (or time) if < 0. */
  ret[0] = (xp->sf_data.SFR < 0.) ? -xp->sf_data.SFR : 0.f;
}

/**
 * @brief Specifies which particle fields to write to a dataset
 *
 * @param parts The particle array.
 * @param xparts The extended data particle array.
 * @param list The list of i/o properties to write.
 * @param with_cosmology Are we running a cosmological simulation?
 *
 * @return Returns the number of fields to write.
 */
__attribute__((always_inline)) INLINE static int star_formation_write_particles(
    const struct part* parts, const struct xpart* xparts, struct io_props* list,
    const int with_cosmology) {

  /* List what we want to write */
  list[0] = io_make_output_field_convert_part(
      "StarFormationRates", FLOAT, 1, UNIT_CONV_SFR, 0.f, parts, xparts,
      convert_part_sfr, "Star formation rates of the particles.");

  if (with_cosmology) {
    list[1] = io_make_output_field_convert_part(
        "LastStarFormationScaleFactors", FLOAT, 1, UNIT_CONV_NO_UNITS, 0.f,
        parts, xparts, convert_part_last_sf_time,
        "Scale-factors at which the stars last had a non-zero star formation "
        "rate.");
  } else {
    list[1] = io_make_output_field_convert_part(
        "LastStarFormationTimes", FLOAT, 1, UNIT_CONV_TIME, 0.f, parts, xparts,
        convert_part_last_sf_time,
        "Times at which the stars last had a non-zero star formation rate.");
  }

  return 2;
}

/**
 * @brief Specifies which sparticle fields to write to a dataset
 *
 * @param sparts The star particle array.
 * @param list The list of i/o properties to write.
 *
 * @return Returns the number of fields to write.
 */
__attribute__((always_inline)) INLINE static int
star_formation_write_sparticles(const struct spart* sparts,
                                struct io_props* list) {
  return 0;
}
#endif /* SWIFT_STAR_FORMATION_EAGLE_IO_H */
