#ifndef SWIFT_DUST_T20_PROPERTIES_H
#define SWIFT_DUST_T20_PROPERTIES_H

/* Config parameters. */
#include <config.h>

/* Standard includes */
#include <hdf5.h>

/* Local includes. */
#include "chemistry_struct.h"
#include "cooling_properties.h"
#include "dust_struct.h"
#include "feedback_properties.h"
#include "physical_constants.h"
#include "units.h"

/**
 * @brief Stores AGB and SNII dust yields
 */

struct dust_yield_table {

  /* Array to read dust yield tables into */
  double* yield;

  /* Array to store IMF-resampled dust yield tables */
  double* yield_IMF_resampled;
};

/**
 * @brief Properties of the dust evolution model.
 */
struct dustevo_props {

  /* ------------ Main operation modes ------------- */

  /*! Are we doing grain destruction by sputtering? */
  int with_sputtering;

  /*! Are we doing grain destruction by SNII? */
  int with_SNII_destruction;

  /*! Are we doing grain growth by accretion? */
  int with_accretion;

  /*! Are we growing small grains by coagulation? */
  int with_coagulation;

  /*! Are we shrinking large grain by shattering? */
  int with_shattering;

  /*! Are we using the subgrid T and rho? */
  int with_subgrid_props;

  /*! Are we pairing the dust fractions to cooling? */
  int pair_to_cooling;

  /*! Are we limiting depletion of elements to prevent over-depletion? */
  int limit_depletion;

  /*! factor by which to scale min diffuse fractions */
  float diff_redfactor;

  /* ------------ Global parameters ------------- */

  /*! Clumping factor assumed for accretion at a given density (default 10.)*/
  float clumping_factor;

  /*! clumping factor to use: 0 => constant clumping factor, 1=> follow cooling
   * clumping factor 2=> compute variable clumping */
  enum clumping_mode clumping_factor_mode;

  /* Density below which the dust clumping factor is
   * equal to unity, in units of cm^-3. */
  float clumping_factor_nH_min_cgs;

  /* Density above which the dust clumping factor is
   * equal to the maximum, in units of cm^-3. */
  float clumping_factor_nH_max_cgs;

  /*! Boost (> 1.) or reduction (< 1.) factor applied to dust diffusion rates
   * (default 1.) */
  float diffusion_rate_boost;

  /*! Dimensionless multiplicative prefactor that can be adjusted to increase or
   * decrease the coagulation timescale normalization
   * (default is 1) */
  float coagulation_timescale_norm_prefactor;

  /*! Fraction of Fe silicate grains (fayalite) as opposed to Mg (forsterite)
   * (default 0.5)*/
  float nu;

  /* ---- Effective silicate properties ---- */

  /* the effective molecular subcript for each element comprising silicate
   * grains */
  float o_sub;
  float mg_sub;
  float si_sub;
  float fe_sub;

  /* ----------- Correcting cooling tables ---------- */

  /* array of element fractions assumed to be in the dust-phase */
  float* logfD;

  /* ------------ Dust yield tables ----------------- */

  /* Yield tables for AGB and SNII  */
  struct dust_yield_table dyield_AGB;
  struct dust_yield_table dyield_SNII;

  /* Path to AGB dust yield table */
  char AGB_dyield_path[200];

  /* ------------- Dust Mapping -------------------- */

  /* metallicity of the Sun */
  float solar_metallicity;

  /* initial abundance of dust species */
  double initial_grain_mass_fraction[dust_grain_species_count];

  /* mass fraction of each grain type constituted by given element */
  double* grain_element_mfrac[dust_grain_species_count];

  /* indices of chemistry array for element contributing to each grain */
  int* grain_element_indices[dust_grain_species_count];

  /* set element count contributing to each grain */
  int grain_element_count[dust_grain_species_count];

  /* set element count contributing to each grain */
  int cgrain_element_count[dust_cgrain_species_count];

  /* mass fraction of each grain type constituted by given element */
  double* cgrain_element_mfrac[dust_cgrain_species_count];

  /* indices of chemistry array for element contributing to each grain */
  int* cgrain_element_indices[dust_cgrain_species_count];

  /* Wiersma solar abundance patterns, for consistency with chemical yields */
  double abundance_pattern[chemistry_element_count];

  /* Element atomic weights */
  double atomic_weight[chemistry_element_count];

  /* Element condensation fractions */
  double condensation_frac[dust_cgrain_species_count];

  /* Grain accretion coeficients (cgs)*/
  double accretion_coeff[dust_cgrain_species_count];

  /* Grain size mass fractions at injection*/
  float grain_size_mass_fraction[dust_grain_species_count];

  /* Grain species size in micron*/
  float grain_size_factors[dust_grain_species_count];

  /* Indices converting between composite and subspecies grain arrays */
  int grain_to_cgrain_indices[dust_grain_species_count];

  /* saturation depletion factors */
  float grain_element_min_diffuse[chemistry_element_count];

  /* mass fraction of composite grain constituted by subspecies  */
  double grain_species_mass_frac[dust_grain_species_count];

  /* --------------- Other Constants --------------- */
  float specific_numSNII;

  /* ratio of atomic weight of Mg over Fe */
  double mg_to_fe_aweight;
};

/**
 * @brief initialise structure housing global dust parametrisation.
 * In particular, flags and values set in the parameter file,
 * and any hard-coded properties
 *
 * @param dp Global dust parameters for initialisation
 * @param params The parsed parameter file.
 * @param phys_const The physical constants in internal units.
 * @param us The current internal system of units.
 */

void dustevo_props_init_backend(struct dustevo_props* dp,
                                struct swift_params* params,
                                struct feedback_props* fp,
                                struct cooling_function_data* cooling,
                                const struct phys_const* phys_const,
                                const struct unit_system* us);

#endif /* SWIFT_DUST_T20_PROPERTIES_H */
