#ifndef SWIFT_DUST_CHEMISTRY_COUPLING_T20_H
#define SWIFT_DUST_CHEMISTRY_COUPLING_T20_H

/* Config parameters. */
#include <config.h>

/* Local includes */
#include "error.h"
#include "part.h"

__attribute__((always_inline)) INLINE static void dust_reset_diffusion(
    struct part* restrict p) {

  struct dust_part_data* cpd = &p->dust_data;

  for (int grain = 0; grain < dust_grain_species_count; ++grain) {
    cpd->dgrain_mass_fraction[grain] = 0.0f;
  }
  cpd->dgrain_iron_mass_fraction_from_SNIa = 0.0f;
}

__attribute__((always_inline)) INLINE static void dust_do_diffusion(
    struct part* p) {

  /* Diffuse each grain individually */
  for (int grain = 0; grain < dust_grain_species_count; ++grain) {
    p->dust_data.grain_mass_fraction[grain] +=
        p->dust_data.dgrain_mass_fraction[grain];
  }
  p->dust_data.grain_iron_mass_fraction_from_SNIa +=
      p->dust_data.dgrain_iron_mass_fraction_from_SNIa;

#ifdef SWIFT_DEBUG_CHECKS
  for (int grain = 0; grain < dust_grain_species_count; ++grain) {
    if (p->dust_data.grain_mass_fraction[grain] !=
        p->dust_data.grain_mass_fraction[grain]) {
      error(
          "NaN dust grain mass fraction for species %i (dgrain_mass_fraction = "
          "%g)!",
          grain, p->dust_data.dgrain_mass_fraction[grain]);
    }
  }
#endif

  for (int i = 0; i < dust_grain_species_count; ++i) {
    if (isnan(p->dust_data.grain_mass_fraction[i]))
      error("Entry i in the dust array is NaN! in function '%s', line %d",
            __FUNCTION__, __LINE__);
  }
}

__attribute__((always_inline)) INLINE static void dust_iact_diffusion(
    struct dust_part_data* dui, struct dust_part_data* duj, const float K_ij,
    const float K_ji, const double dt) {

  /* Compute contribution to the dust abundance */
  for (int i = 0; i < dust_grain_species_count; i++) {
    dui->dgrain_mass_fraction[i] +=
        K_ij * (dui->grain_mass_fraction[i] - duj->grain_mass_fraction[i]) * dt;
    duj->dgrain_mass_fraction[i] +=
        K_ji * (duj->grain_mass_fraction[i] - dui->grain_mass_fraction[i]) * dt;
  }

  dui->dgrain_iron_mass_fraction_from_SNIa +=
      K_ij *
      (dui->grain_iron_mass_fraction_from_SNIa -
       duj->grain_iron_mass_fraction_from_SNIa) *
      dt;

  duj->dgrain_iron_mass_fraction_from_SNIa +=
      K_ji *
      (duj->grain_iron_mass_fraction_from_SNIa -
       dui->grain_iron_mass_fraction_from_SNIa) *
      dt;
}

__attribute__((always_inline)) INLINE static void dust_iact_nonsym_diffusion(
    struct dust_part_data* dui, const struct dust_part_data* duj,
    const float K_ij, const double dt) {

  /* Compute contribution to the dust abundance */
  for (int i = 0; i < dust_grain_species_count; i++) {
    dui->dgrain_mass_fraction[i] +=
        K_ij * (dui->grain_mass_fraction[i] - duj->grain_mass_fraction[i]) * dt;
  }

  dui->dgrain_iron_mass_fraction_from_SNIa +=
      K_ij *
      (dui->grain_iron_mass_fraction_from_SNIa -
       duj->grain_iron_mass_fraction_from_SNIa) *
      dt;
}

#endif /* SWIFT_DUST_CHEMISTRY_COUPLING_T20_H */
