#ifndef SWIFT_DUST_IO_NONE_H
#define SWIFT_DUST_IO_NONE_H

#include "chemistry.h"
#include "dust.h"
#include "io_properties.h"

/**
 * @brief Specifies which particle fields to write to a dataset
 *
 * No output to add
 *
 * @param parts The particle array.
 * @param list The list of i/o properties to write.
 *
 * @return Returns the number of fields to write.
 */
INLINE static int dust_write_particles(const struct part* parts,
                                       const struct xpart* xparts,
                                       struct io_props* list,
                                       const int with_cosmology) {
  return 0;
}

/**
 * @brief Specifies which black hole particle fields to write to a dataset
 *
 * No output to add.
 *
 * @param bparts The black hole particle array.
 * @param list The list of i/o properties to write.
 *
 * @return Returns the number of fields to write.
 */
INLINE static int dust_write_bparticles(const struct bpart* bparts,
                                        struct io_props* list) {
  return 0;
}

/**
 * @brief Writes the current model of SPH to the file
 *
 * Nothing here.
 *
 * @param h_grp The HDF5 group in which to write
 * @param h_grp_columns The HDF5 group containing named columns
 * @param e The #engine.
 */
INLINE static void dust_write_flavour(hid_t h_grp, hid_t h_grp_columns,
                                      const struct engine* e) {}

/**
 * @brief Create and write array for mapping elements to dust grains
 *
 * Nothing here.
 *
 * @param h_grp The HDF5 group in which to write
 * @param e The #engine.
 */
INLINE static void dust_write_composition(hid_t h_grp, const struct engine* e) {
}

/**
 * @brief Conversion function for adding explicit depletion back to metal arrays
 *
 * for "none" dust model, just write out metal mass fractions normally.
 *
 * @param e The #engine.
 * @param p The particle array.
 * @param xp The extended particle array.
 * @param ret Output array.
 */
INLINE static void convert_part_add_back_dust_for_chemistry(
    const struct engine* e, const struct part* p, const struct xpart* xp,
    float* ret) {

#if defined(CHEMISTRY_EAGLE) || defined(CHEMISTRY_COLIBRE)
  for (int i = 0; i < chemistry_element_count; i++)
    ret[i] = (float)p->chemistry_data.metal_mass_fraction[i];
#endif
}

INLINE static void convert_part_add_back_dust_for_reduced_chemistry(
    const struct engine* e, const struct part* p, const struct xpart* xp,
    float* ret) {

#if defined(CHEMISTRY_EAGLE) || defined(CHEMISTRY_COLIBRE)
  ret[0] = (float)p->chemistry_data.metal_mass_fraction[chemistry_element_H];
  ret[1] = (float)p->chemistry_data.metal_mass_fraction[chemistry_element_He];
#else
  ret[0] = 0.;
  ret[1] = 0.;
#endif
}

/**
 * @brief Conversion function for adding back depleted SNIa iron
 *
 * for "none" dust model, just write out SNIa iron normally.
 *
 * @param e The #engine.
 * @param p The particle array.
 * @param xp The extended particle array.
 * @param ret Output array.
 */
INLINE static void convert_part_add_back_dust_for_snia_iron(
    const struct engine* e, const struct part* p, const struct xpart* xp,
    float* ret) {
#if defined(CHEMISTRY_EAGLE) || defined(CHEMISTRY_COLIBRE)
  *ret = p->chemistry_data.iron_mass_fraction_from_SNIa;
#endif
}

/**
 * @brief Conversion function for adding explicit depletion back to metal arrays
 *
 * for "none" dust model, just write out metal mass fractions normally.
 *
 * @param e The #engine.
 * @param bp The black hole particle array.
 * @param ret Output array.
 */
INLINE static void convert_bpart_add_back_dust_for_chemistry(
    const struct engine* e, const struct bpart* bp, float* ret) {
#if defined(CHEMISTRY_EAGLE) || defined(CHEMISTRY_COLIBRE)
  for (int i = 0; i < chemistry_element_count; i++)
    ret[i] = (float)bp->chemistry_data.metal_mass[i];
#endif
}

/**
 * @brief Conversion function for adding back depleted SNIa iron
 *
 * for "none" dust model, just write out SNIa iron normally.
 *
 * @param e The #engine.
 * @param bp The black hole particle array.
 * @param ret Output array.
 */
INLINE static void convert_bpart_add_back_dust_for_snia_iron(
    const struct engine* e, const struct bpart* bp, float* ret) {
#if defined(CHEMISTRY_EAGLE) || defined(CHEMISTRY_COLIBRE)
  *ret = bp->chemistry_data.iron_mass_from_SNIa;
#endif
}

#endif /* SWIFT_DUST_NONE_PROPERTIES_H */
