/*******************************************************************************
 * This file is part of SWIFT.
 * Copyright (c) 2016 Matthieu Schaller (matthieu.schaller@durham.ac.uk)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#ifndef SWIFT_DUST_CONSTANTS_H
#define SWIFT_DUST_CONSTANTS_H

/* Config parameters. */
#include <config.h>

/* Dust-to-gas ratio representing the solar neighbourhood, matching value
 * assumed internally by CHIMES.*/
static const float local_dust_to_gas_ratio = 0.006;

/* Solar metallicity following  Wiersma (2009a) enrichment tables */
static const float solar_metallicity_wiersma = 0.0127;

/* Typical grain radius in cm, consistent with AGB and SNe observations
 * (e.g. Groenewegen 1997, Nozawa 2007) */
static const float representative_grain_size = 1e-5;

/* Sputtering timescale normalisation in 1/s, following Tsai & Matthews (1995)
 */
static const double sputtering_timescale_norm = 3.2e-18;

/* Fraction of zero age mass in stars above the SNII threshold for a
 * Chabrier (2000) IMF */
static const float SNII_mass_fraction = 7.039463e-3;

/* Normalisation of gas mass swept by SNII shocks in from Yamasawa et. al
 * (2011), 1000 solar masses */
static const double snii_destruction_coeff = 3.052e+36;

/* ===============================
 * ======== T20 specific =========
 * =============================== */

/* Solar abundance pattern, assuming Wiersma et al (2009a) */
static const float solar_abundance_H = 7.0649785e-01;
static const float solar_abundance_He = 2.8055534e-01;
static const float solar_abundance_C = 2.0665436e-03;
static const float solar_abundance_N = 8.3562563e-04;
static const float solar_abundance_O = 5.4926244e-03;
static const float solar_abundance_Ne = 1.4144605e-03;
static const float solar_abundance_Mg = 5.9070642e-04;
static const float solar_abundance_Si = 6.8258739e-04;
static const float solar_abundance_Fe = 1.1032152e-03;

/* Atomic weights for traced element up to Fe */
static const float atomic_weight_H = 1.0079;
static const float atomic_weight_He = 4.0026;
static const float atomic_weight_C = 12.0107;
static const float atomic_weight_N = 14.0067;
static const float atomic_weight_O = 15.9994;
static const float atomic_weight_Ne = 20.1797;
static const float atomic_weight_Mg = 24.305;
static const float atomic_weight_Si = 28.0855;
static const float atomic_weight_Fe = 55.845;

/* Define the fiducial diffuse (i.e. complement of fiducial depleted)
 * fractions of elements constituting dust. These are taken from the
 * table dust assumptions for MW-like ISM conditions, and are scaled
 * by some factor to put a limit on depletion */
static const float fiducial_diffuse_C = 0.34385;
static const float fiducial_diffuse_O = 0.31766;
static const float fiducial_diffuse_Mg = 0.94338;
static const float fiducial_diffuse_Si = 0.94492;
static const float fiducial_diffuse_Fe = 0.99363;

/* values determining the number of elements in ech effective grain
 * chemistry */
static const int atomicity_graphite = 1;
static const int atomicity_silicate = 4;
static const int atomicity_mgsilicate = 3;
static const int atomicity_fesilicate = 3;

/* Accretion timescale normalisation in seconds. Taken from the
 * default values of Hirashita & Voschinnikov (2013) */
static const float graphite_accretion_timescale_norm = 3.132e15;
static const float silicate_accretion_timescale_norm = 5.677e15;

/* Unitless condensation efficiencies for grains out of SNII outflow,
 * taken from Zhukovska et al (2008) */
static const float graphite_condensation_efficiency = 0.15;
static const float silicate_condensation_efficiency = 3.5e-4;

/* Shattering timescale normalisation in seconds. Taken from the
 * default values of Aoyama et al (2017, App. A.) */
static const double shattering_timescale_norm = 1.706e15;

/* Coagulation timescale normalisation derived from Yan et al.
 * (2004), in s. */
static const double coagulation_timescale_norm = 1.709e13;  // 4.273e14
// static const double  coagulation_velocity_norm = 2e4;

/* radius scaling factor relative to the fiducial 0.1 micron value */
static const float radius_scaling_large = 1.;
static const float radius_scaling_small = 0.1;

/* ------------ Parameter Defaults ------------ */

/* Default clumping factor */
static const float default_clumping_factor = 10.;

/* Default number fraction of iron (as opposed to magnesium) grains */
static const float default_fe_grain_fraction = 0.5;

/* Default fractions of injected grain sizes*/
static const float default_injection_fraction_large = 0.9;
static const float default_injection_fraction_small = 0.1;

/* Default silicate composition (olivine-like) */
static const float default_subscript_silicate_O = 6.;
static const float default_subscript_silicate_Mg = 1.;
static const float default_subscript_silicate_Si = 2.;
static const float default_subscript_silicate_Fe = 1.;

/* Factor to scale fiducial diffuse fractions by to limit depletion */
static const float default_diffuse_fraction_scaling = 0.423345;

#endif /* SWIFT_DUST_CONSTANTS_H */
