#ifndef SWIFT_DUST_M16_H
#define SWIFT_DUST_M16_H

/* Config parameters. */
#include <config.h>

/* Local includes */
#include "dust_properties.h"

struct phys_const;
struct unit_system;
struct hydro_props;
struct entropy_floor_properties;
struct cooling_function_data;

void dust_redistribute_masses_astration(struct spart* sp, const struct part* p,
                                        const struct dustevo_props* dp);

static INLINE void dust_redistribute_masses_cooling_step(
    struct part* p, const struct dustevo_props* dp) {}

void dustevo_print_backend(const struct dustevo_props* dp);

void dust_evolve_part(const struct phys_const* phys_const,
                      const struct unit_system* us,
                      const struct cosmology* cosmo,
                      const struct hydro_props* hydro_properties,
                      const struct entropy_floor_properties* floor_props,
                      const struct cooling_function_data* cooling,
                      const struct dustevo_props* dp, struct part* restrict p,
                      struct xpart* restrict xp, const float dt,
                      const float dt_therm, const double time);

void dustevo_props_dump(const struct dustevo_props* dp, FILE* stream);

void dustevo_props_restore(struct dustevo_props* dp, struct feedback_props* fp,
                           FILE* stream);

void dustevo_set_jagged_arrays(struct dustevo_props* dp);

/**
 * @brief Return a string containing the name of a given #grain_species.
 */
__attribute__((always_inline)) INLINE static const char* dust_get_grain_name(
    enum grain_species grain) {

  static const char* dust_grain_species_names[dust_grain_species_count] = {
      "DepletedCarbon", "DepletedOxygen", "DepletedMagnesium",
      "DepletedSilicon", "DepletedIron" /*, DepletedUranium*/};

  return dust_grain_species_names[grain];
}

#ifdef HAVE_HDF5
void dust_read_depletion(hid_t id, float** log_depletion_fractions,
                         const int table_cooling_N_redshifts,
                         const int table_cooling_N_temperature,
                         const int table_cooling_N_metallicity,
                         const int table_cooling_N_density,
                         const int table_cooling_N_elementtypes);
#endif

/**
 * @brief Prepares a particle for the smooth dust calculation.
 *
 * Zeroes all the relevant arrays in preparation for the sums taking place in
 * the various smooth dust tasks
 *
 * @param p The particle to act upon
 * @param dp #chemistry_global_data containing chemistry informations.
 */
__attribute__((always_inline)) INLINE static void dust_init_part(
    struct part* restrict p, const struct dustevo_props* dp) {

  struct dust_part_data* cpd = &p->dust_data;

  for (int grain = 0; grain < dust_grain_species_count; ++grain) {
    cpd->dgrain_mass_fraction[grain] = 0.0f;
  }
}

__attribute__((always_inline)) INLINE static void dust_first_init_part(
    const struct phys_const* restrict phys_const,
    const struct unit_system* restrict us,
    const struct cosmology* restrict cosmo, const struct dustevo_props* dp,
    struct part* restrict p, struct xpart* restrict xp) {

  // Add initialization of fields in dust_part_data struct.
  for (int grain = 0; grain < dust_grain_species_count; ++grain) {
    p->dust_data.grain_mass_fraction[grain] =
        dp->initial_grain_mass_fraction[grain];
  }
  dust_init_part(p, dp);
}

/**
 * @brief Initialise the dust properties of a black hole with
 * the dust properties of the gas it is born from.
 *
 * Black holes don't store fractions so we need to use dust masses.
 *
 * @param bp_data The black hole data to initialise.
 * @param p_data The gas data to use.
 * @param gas_mass The mass of the gas particle.
 */
__attribute__((always_inline)) INLINE static void dust_bpart_from_part(
    struct dust_bpart_data* bp_data, const struct dust_part_data* p_data,
    const double gas_mass) {

  for (int i = 0; i < dust_grain_species_count; ++i) {
    bp_data->grain_mass[i] = p_data->grain_mass_fraction[i] * gas_mass;
  }
}

/**
 * @brief Add the dust data of a gas particle to a black hole.
 *
 * Black holes don't store fractions so we need to add dust masses.
 *
 * @param bp_data The black hole data to add to.
 * @param p_data The gas data to use.
 * @param gas_mass The mass of the gas particle.
 */
__attribute__((always_inline)) INLINE static void dust_add_part_to_bpart(
    struct dust_bpart_data* bp_data, const struct dust_part_data* p_data,
    const double gas_mass) {

  for (int i = 0; i < dust_grain_species_count; ++i) {
    bp_data->grain_mass[i] += p_data->grain_mass_fraction[i] * gas_mass;
  }
}

/**
 * @brief Transfer dust data of a gas particle to a black hole.
 *
 * In contrast to `dust_add_part_to_bpart`, only a fraction of the
 * masses stored in the gas particle are transferred here. Absolute masses
 * of the gas particle are adjusted as well.
 * Black holes don't store fractions so we need to add dust masses.
 *
 * @param bp_data The black hole data to add to.
 * @param p_data The gas data to use.
 * @param nibble_mass The mass to be removed from the gas particle.
 * @param nibble_fraction The fraction of the (original) mass of the gas
 *        particle that is removed.
 */
__attribute__((always_inline)) INLINE static void dust_transfer_part_to_bpart(
    struct dust_bpart_data* bp_data, struct dust_part_data* p_data,
    const double nibble_mass, const double nibble_fraction) {

  for (int i = 0; i < dust_grain_species_count; ++i)
    bp_data->grain_mass[i] += p_data->grain_mass_fraction[i] * nibble_mass;
}

/**
 * @brief Add the dust data of a black hole to another one.
 *
 * @param bp_data The black hole data to add to.
 * @param swallowed_data The black hole data to use.
 */
__attribute__((always_inline)) INLINE static void dust_add_bpart_to_bpart(
    struct dust_bpart_data* bp_data,
    const struct dust_bpart_data* swallowed_data) {

  for (int i = 0; i < dust_grain_species_count; ++i) {
    bp_data->grain_mass[i] += swallowed_data->grain_mass[i];
  }
}

#endif /* SWIFT_DUST_M16_PROPERTIES_H */
